/*
 * Decompiled with CFR 0.152.
 */
package com.gridnine.xtrip.common.model.dict;

import com.gridnine.xtrip.common.Environment;
import com.gridnine.xtrip.common.meta.DictionaryType;
import com.gridnine.xtrip.common.meta.MetaRegistry;
import com.gridnine.xtrip.common.model.dict.BaseDictionary;
import com.gridnine.xtrip.common.model.dict.CacheConfig;
import com.gridnine.xtrip.common.model.dict.CacheData;
import com.gridnine.xtrip.common.model.dict.DictSearchCriterion;
import com.gridnine.xtrip.common.model.dict.DictionaryReference;
import com.gridnine.xtrip.common.model.dict.Index;
import com.gridnine.xtrip.common.service.ExecutorServiceFacade;
import com.gridnine.xtrip.common.util.IoUtil;
import com.gridnine.xtrip.common.util.MiscUtil;
import com.gridnine.xtrip.common.util.TextUtil;
import java.io.BufferedInputStream;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.time.LocalDateTime;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.function.Consumer;
import java.util.stream.Collectors;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

final class DefaultCacheData
implements CacheData {
    private static final Logger log = LoggerFactory.getLogger(DefaultCacheData.class);
    private volatile CacheConfig config = new CacheConfig();
    private final ConcurrentMap<String, IndexData> indexes = new ConcurrentHashMap<String, IndexData>();
    private final Set<String> invalidDictionaries = new CopyOnWriteArraySet<String>();
    private final Set<String> ignoredDictionaryTypes = new HashSet<String>();
    private final long debouncePeriod;

    static void load(File folder, DefaultCacheData data) {
        if (log.isDebugEnabled()) {
            log.debug(String.format("load [folder = %s, data = %s]", folder != null ? folder.getPath() : "null", String.valueOf(data)));
            data.dump();
        }
        if (folder == null || !folder.isDirectory()) {
            return;
        }
        long timing = System.currentTimeMillis();
        File configFile = new File(folder, MiscUtil.getSimpleClassName(CacheConfig.class) + ".ser");
        if (!configFile.exists()) {
            log.warn("cache config file does not exist");
            IoUtil.emptyFolder(folder);
            return;
        }
        try (ObjectInputStream strm = new ObjectInputStream(new BufferedInputStream(Files.newInputStream(configFile.toPath(), new OpenOption[0])));){
            CacheConfig config = (CacheConfig)strm.readObject();
            log.debug("cache config loaded from " + configFile + ". " + config.toString());
            data.config = config;
        }
        catch (Exception e2) {
            log.warn("failed loading cache config from file " + configFile, (Throwable)e2);
            IoUtil.emptyFolder(folder);
            data.reset(null);
            return;
        }
        if (data.getLastCheck() != null && MiscUtil.toLocalDateTime(data.getLastCheck()).plusMonths(3L).isBefore(LocalDateTime.now())) {
            log.info("dictionary cache is hopelessly outdated, clear it");
            IoUtil.emptyFolder(folder);
            data.reset(null);
            return;
        }
        Consumer<String> loader = clsName -> {
            File indexFile = new File(folder, clsName + ".ser");
            if (!indexFile.isFile()) {
                log.warn(String.format("index data file %s does not exist", indexFile));
                data.resetDictionary((String)clsName);
                return;
            }
            try (ObjectInputStream strm = new ObjectInputStream(new BufferedInputStream(Files.newInputStream(indexFile.toPath(), new OpenOption[0])));){
                Index index = (Index)strm.readObject();
                log.debug("cache index loaded from " + indexFile);
                data.indexes.put((String)clsName, new IndexData(index));
            }
            catch (Exception e) {
                log.warn("failed loading cache index from file " + indexFile, (Throwable)e);
                data.resetDictionary((String)clsName);
                try {
                    if (!indexFile.delete()) {
                        throw new IOException("unable to delete file " + indexFile);
                    }
                }
                catch (Exception e2) {
                    log.warn("failed loading cache index from file " + indexFile, (Throwable)e2);
                }
            }
        };
        List<String> names = MetaRegistry.get().getDictionaries().entrySet().stream().filter(e -> !data.ignoredDictionaryTypes.contains(e.getKey())).peek(e -> {
            if (((DictionaryType)e.getValue()).isReset()) {
                data.resetDictionary((String)e.getKey());
            }
        }).filter(e -> !((DictionaryType)e.getValue()).isReset()).map(Map.Entry::getKey).collect(Collectors.toList());
        int threadCount = Integer.getInteger("com.gridnine.xtrip.common.cache-data.load.thread-count", 1);
        if (threadCount < 2) {
            names.forEach(loader);
        } else {
            try (ExecutorServiceFacade service = Environment.getPublished(ExecutorServiceFacade.class).newFixedThreadPool("load-cache-data", threadCount);){
                CountDownLatch barrier = new CountDownLatch(names.size());
                for (String clsName2 : names) {
                    service.submit(() -> {
                        try {
                            loader.accept(clsName2);
                        }
                        finally {
                            barrier.countDown();
                        }
                    });
                }
                while (true) {
                    try {
                        barrier.await();
                    }
                    catch (InterruptedException interruptedException) {
                        continue;
                    }
                    break;
                }
            }
        }
        if (log.isDebugEnabled()) {
            data.dump();
        }
        MiscUtil.logTiming(timing, "cache data loaded");
    }

    static boolean canSave(DefaultCacheData data) {
        long currentTimeMillis = System.currentTimeMillis();
        for (Map.Entry entry : data.indexes.entrySet()) {
            IndexData index = (IndexData)entry.getValue();
            if (index.getIndex().getModified() == null || currentTimeMillis - index.getIndex().getModified().getTime() >= data.debouncePeriod) continue;
            return false;
        }
        return true;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    static void save(File file, DefaultCacheData data) {
        if (log.isDebugEnabled()) {
            log.debug(String.format("save [file = %s, data = %s]", file != null ? file.getPath() : "null", String.valueOf(data)));
            if (data != null) {
                data.dump();
            }
        }
        if (file == null || data == null) {
            return;
        }
        long timing = System.currentTimeMillis();
        file.mkdirs();
        File configFile = new File(file, MiscUtil.getSimpleClassName(CacheConfig.class) + ".ser");
        if (configFile.exists() && !configFile.delete()) {
            log.warn("failed deleting file " + file);
            return;
        }
        for (Map.Entry entry : data.indexes.entrySet()) {
            IndexData index = (IndexData)entry.getValue();
            Lock lock = index.getLock();
            lock.lock();
            try {
                if (index.getIndex().getModified() == null) continue;
                File indexFile = new File(file, (String)entry.getKey() + ".ser");
                if (indexFile.exists() && !indexFile.delete()) {
                    log.warn("failed deleting file " + indexFile);
                    return;
                }
                try {
                    ObjectOutputStream strm = new ObjectOutputStream(new BufferedOutputStream(Files.newOutputStream(indexFile.toPath(), new OpenOption[0])));
                    Throwable throwable = null;
                    try {
                        strm.writeObject(index.getIndex());
                        index.getIndex().setModified(null);
                        log.debug("cache index wrote to " + indexFile);
                    }
                    catch (Throwable throwable2) {
                        throwable = throwable2;
                        throw throwable2;
                    }
                    finally {
                        if (strm == null) continue;
                        if (throwable != null) {
                            try {
                                strm.close();
                            }
                            catch (Throwable throwable3) {
                                throwable.addSuppressed(throwable3);
                            }
                            continue;
                        }
                        strm.close();
                    }
                }
                catch (Exception e) {
                    log.warn("failed writing index to " + indexFile, (Throwable)e);
                    lock.unlock();
                    return;
                }
            }
            finally {
                lock.unlock();
            }
        }
        try (ObjectOutputStream strm = new ObjectOutputStream(new BufferedOutputStream(Files.newOutputStream(configFile.toPath(), new OpenOption[0])));){
            strm.writeObject(data.config);
            log.debug("cache config wrote to " + configFile);
        }
        catch (Exception e) {
            log.warn("failed writing config to " + configFile, (Throwable)e);
            return;
        }
        MiscUtil.logTiming(timing, "cache data saved");
    }

    DefaultCacheData(Set<String> ignoredTypes) {
        this.ignoredDictionaryTypes.addAll(ignoredTypes);
        String debouncePeriodStr = System.getProperty("midoffice.debouncePeriod");
        this.debouncePeriod = !TextUtil.isBlank(debouncePeriodStr) ? TimeUnit.SECONDS.toMillis(Integer.parseInt(debouncePeriodStr)) : TimeUnit.MINUTES.toMillis(5L);
    }

    @Override
    public Date getLastCheck() {
        return this.config.getLastCheck();
    }

    @Override
    public void setLastCheck(Date value) {
        this.config.setLastCheck(value);
        this.invalidDictionaries.clear();
    }

    @Override
    public String getVersion() {
        return this.config.getVersion();
    }

    @Override
    public void reset(String version) {
        this.config.setLastCheck(null);
        this.config.setVersion(version);
        this.indexes.clear();
        this.invalidDictionaries.clear();
    }

    void resetDictionary(String className) {
        this.invalidDictionaries.add(className);
        this.remove(className);
    }

    @Override
    public Set<String> getInvalidDictionaries() {
        return this.invalidDictionaries;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void add(BaseDictionary dict) {
        IndexData index = (IndexData)this.indexes.get(dict.getClass().getName());
        if (index == null) {
            index = new IndexData(new Index());
            IndexData prev = this.indexes.putIfAbsent(dict.getClass().getName(), index);
            if (prev != null) {
                index = prev;
            }
        }
        Lock lock = index.getLock();
        lock.lock();
        try {
            index.getIndex().add(dict, true);
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void remove(String className, String code) {
        IndexData index = (IndexData)this.indexes.get(className);
        if (index != null) {
            Lock lock = index.getLock();
            lock.lock();
            try {
                index.getIndex().remove(code, true);
                if (index.getIndex().size() == 0) {
                    this.remove(className);
                }
            }
            finally {
                lock.unlock();
            }
        }
    }

    @Override
    public void remove(String dictionaryClassName) {
        this.indexes.put(dictionaryClassName, new IndexData(new Index()));
    }

    @Override
    public void remove(DictionaryReference<?> ref) {
        this.remove(ref.getType().getName(), ref.getCode());
    }

    @Override
    public void remove(BaseDictionary dict) {
        this.remove(dict.getClass().getName(), dict.getCode());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <D extends BaseDictionary> Map<String, D> getAll(Class<D> cls) {
        IndexData index = (IndexData)this.indexes.get(cls.getName());
        if (index == null) {
            return Collections.emptyMap();
        }
        Map<String, ?> snapshot = index.getIndex().getSnapshot();
        if (snapshot != null) {
            return snapshot;
        }
        Lock lock = index.getLock();
        lock.lock();
        try {
            Map<String, ?> map = index.getIndex().createNGetSnapshot();
            return map;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <D extends BaseDictionary> D findByCode(Class<D> cls, String code) {
        IndexData index = (IndexData)this.indexes.get(cls.getName());
        if (index == null) {
            return null;
        }
        Lock lock = index.getLock();
        lock.lock();
        try {
            Object obj = index.getIndex().findByCode(code);
            return (D)obj;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <D extends BaseDictionary> D findByRemoteUid(Class<D> cls, String remoteUid) {
        IndexData index = (IndexData)this.indexes.get(cls.getName());
        if (index == null) {
            return null;
        }
        Lock lock = index.getLock();
        lock.lock();
        try {
            Object obj = index.getIndex().findByRemoteUid(remoteUid);
            return (D)obj;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <D extends BaseDictionary> Set<D> lookup(Class<D> cls, String code, String ... codeSystems) {
        IndexData index = (IndexData)this.indexes.get(cls.getName());
        if (index == null) {
            return Collections.emptySet();
        }
        Lock lock = index.getLock();
        lock.lock();
        try {
            Set<?> set = index.getIndex().lookup(code, codeSystems);
            return set;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <D extends BaseDictionary> Set<D> search(Class<D> cls, String pattern) {
        IndexData index = (IndexData)this.indexes.get(cls.getName());
        if (index == null) {
            return Collections.emptySet();
        }
        Lock lock = index.getLock();
        lock.lock();
        try {
            Set<?> set = index.getIndex().search(pattern);
            return set;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public <D extends BaseDictionary> Set<D> search(Class<D> cls, DictSearchCriterion criterion) {
        IndexData index = (IndexData)this.indexes.get(cls.getName());
        if (index == null) {
            return Collections.emptySet();
        }
        Lock lock = index.getLock();
        lock.lock();
        try {
            Set<?> set = index.getIndex().search(criterion);
            return set;
        }
        finally {
            lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void dump() {
        if (!log.isDebugEnabled()) {
            return;
        }
        StringBuilder buf = new StringBuilder();
        for (Map.Entry entry : this.indexes.entrySet()) {
            buf.append("\t").append((String)entry.getKey()).append("\r\n");
            Lock lock = ((IndexData)entry.getValue()).getLock();
            lock.lock();
            try {
                ((IndexData)entry.getValue()).getIndex().dump(buf);
            }
            finally {
                lock.unlock();
            }
        }
        buf.append("\tLast check: ").append(this.config.getLastCheck());
        log.debug("cache state dump:\r\n" + buf);
    }

    @Override
    public void dispose() {
        this.indexes.clear();
        this.config.setLastCheck(null);
    }

    @Override
    public void loadData(File file, CacheData data) {
        DefaultCacheData.load(file, (DefaultCacheData)data);
    }

    @Override
    public void saveData(File file, CacheData data) {
        DefaultCacheData.save(file, (DefaultCacheData)data);
    }

    @Override
    public boolean canSave(CacheData data) {
        return DefaultCacheData.canSave((DefaultCacheData)data);
    }

    private static class IndexData {
        private final Index<?> index;
        private final Lock lock = new ReentrantLock();

        public IndexData(Index<?> idx) {
            this.index = idx;
        }

        public Index<?> getIndex() {
            return this.index;
        }

        public Lock getLock() {
            return this.lock;
        }
    }
}

