/*
 * Decompiled with CFR 0.152.
 */
package com.gridnine.xtrip.server.logging;

import com.google.gson.stream.JsonWriter;
import com.gridnine.xtrip.common.Environment;
import com.gridnine.xtrip.common.util.TextUtil;
import com.gridnine.xtrip.common.xml.XHelper;
import com.gridnine.xtrip.server.logging.LogInfoEnricher;
import com.gridnine.xtrip.server.logging.LogServerHartbeatMessage;
import com.gridnine.xtrip.server.logging.LogServerMessage;
import com.gridnine.xtrip.server.logging.LogServerMessageBase;
import com.gridnine.xtrip.server.logging.LogServerSettings;
import com.gridnine.xtrip.server.logging.TrustAllTrustManager;
import java.io.BufferedOutputStream;
import java.io.BufferedWriter;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Writer;
import java.lang.ref.SoftReference;
import java.net.URL;
import java.nio.channels.Channels;
import java.nio.channels.FileChannel;
import java.nio.channels.WritableByteChannel;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.FileAttribute;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.Iterator;
import java.util.Objects;
import java.util.Queue;
import java.util.UUID;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.zip.GZIPOutputStream;
import javax.net.ssl.HttpsURLConnection;
import javax.net.ssl.SSLContext;
import javax.net.ssl.TrustManager;
import org.apache.log4j.helpers.LogLog;
import org.slf4j.LoggerFactory;

class LogServerConnector {
    private static final long MESSAGE_CHECK_INTERVAL_MS = 15000L;
    private static final long TIME_BEFORE_SWITCH_TO_FILE_MS = 60000L;
    private static final long TIME_BEFORE_NEXT_TRY_TO_SEND_MS = 120000L;
    private static final int HTTP_TIMEOUT_CONNECT_MS = 10000;
    private static final int HTTP_TIMEOUT_READ_MS = 60000;
    private static final int HARTBEAT_INTERVAL_MS = 300000;
    private final Thread senderThread;
    final LogServerSettings serverSettings;
    private final Queue<SoftReference<LogServerMessage>> messagesQueue = new ConcurrentLinkedQueue<SoftReference<LogServerMessage>>();
    private long connectionErrorStarted = -1L;
    private boolean backupFileExists = false;
    private final Path backupFile = Environment.getTempFolder().toPath().resolve("log_server_buffer.tmp");
    private final TrustManager[] trustAllCerts;
    private final LogInfoEnricher logInfoEnricher;

    LogServerConnector(LogServerSettings serverSettings) {
        this.serverSettings = serverSettings;
        LogInfoEnricher enr = null;
        String enricherClass = System.getProperty("loginfo-enricher.className");
        if (!TextUtil.isBlank((String)enricherClass)) {
            try {
                enr = (LogInfoEnricher)XHelper.getClass((String)enricherClass).newInstance();
            }
            catch (Exception e) {
                LoggerFactory.getLogger(LogServerConnector.class).error("unable to init " + enricherClass);
            }
        }
        this.logInfoEnricher = enr;
        if (Files.exists(this.backupFile, new LinkOption[0])) {
            this.backupFileExists = true;
        }
        if (this.serverSettings == null) {
            this.trustAllCerts = null;
            this.senderThread = null;
            LogLog.warn((String)"Log server connector initialized in offline mode");
            return;
        }
        this.trustAllCerts = serverSettings.allowSelfSignedCertificate ? new TrustManager[]{new TrustAllTrustManager()} : null;
        this.senderThread = new Thread(){
            private final Collection<SoftReference<LogServerMessage>> messagesBuffer = new ArrayList<SoftReference<LogServerMessage>>();
            private long lastHartbeatAttempt = -1L;

            @Override
            public void run() {
                while (true) {
                    Object msg;
                    if (System.currentTimeMillis() - this.lastHartbeatAttempt >= 300000L) {
                        try {
                            LogServerConnector.this.sendDataToServer(out -> {
                                try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)out, StandardCharsets.UTF_8);){
                                    LogServerConnector.this.writeMessagesToJson(writer, Collections.singleton(new LogServerHartbeatMessage()));
                                }
                            });
                            this.lastHartbeatAttempt = System.currentTimeMillis();
                        }
                        catch (Exception e) {
                            msg = "Error sending heartbeat message: ";
                            LoggerFactory.getLogger(LogServerConnector.class).error((String)msg + e);
                            LogLog.error((String)msg, (Throwable)e);
                        }
                    }
                    if (!LogServerConnector.this.messagesQueue.isEmpty()) {
                        SoftReference elt = null;
                        while ((elt = (SoftReference)LogServerConnector.this.messagesQueue.poll()) != null) {
                            msg = (LogServerMessage)elt.get();
                            if (msg == null) continue;
                            this.messagesBuffer.add(elt);
                        }
                        if (!this.messagesBuffer.isEmpty()) {
                            ArrayList<LogServerMessage> localBuffer = new ArrayList<LogServerMessage>(this.messagesBuffer.size());
                            Iterator<SoftReference<LogServerMessage>> it = this.messagesBuffer.iterator();
                            while (it.hasNext()) {
                                SoftReference<LogServerMessage> elt2 = it.next();
                                LogServerMessage msg2 = elt2.get();
                                if (msg2 == null) {
                                    it.remove();
                                    continue;
                                }
                                localBuffer.add(msg2);
                            }
                            for (LogServerMessage message : localBuffer) {
                                if (message.getMessageId() == null) {
                                    message.setMessageId(LogServerConnector.this.generateMessageId());
                                }
                                if (message.hasOrganizationInfo() || LogServerConnector.this.logInfoEnricher == null) continue;
                                LogServerConnector.this.logInfoEnricher.addOrganizationInfo(message);
                            }
                            boolean messagesSent = LogServerConnector.this.trySendMessagesToServer(localBuffer);
                            if (messagesSent) {
                                this.messagesBuffer.clear();
                            }
                        }
                    }
                    if (this.isInterrupted()) {
                        return;
                    }
                    try {
                        Thread.sleep(15000L);
                    }
                    catch (InterruptedException e) {
                        return;
                    }
                }
            }
        };
        this.senderThread.setName("ELK-Sender-Thread");
        this.senderThread.setDaemon(true);
        this.senderThread.start();
        LogLog.warn((String)"Log server connector initialized in online mode");
    }

    void submitMessage(LogServerMessage message) {
        if (this.serverSettings == null) {
            if (message.getMessageId() == null) {
                message.setMessageId(this.generateMessageId());
            }
            this.writeMessagesToBackupFile(Collections.singleton(message));
            return;
        }
        this.messagesQueue.add(new SoftReference<LogServerMessage>(message));
    }

    private boolean trySendMessagesToServer(Collection<LogServerMessage> messages) {
        long now = System.currentTimeMillis();
        if (this.connectionErrorStarted != -1L && now - this.connectionErrorStarted >= 60000L) {
            if (now - this.connectionErrorStarted >= 120000L) {
                this.connectionErrorStarted = -1L;
            } else {
                return this.writeMessagesToBackupFile(messages);
            }
        }
        return this.sendMessagesToServer(messages);
    }

    private boolean sendMessagesToServer(Collection<LogServerMessage> messages) {
        LogLog.debug((String)("Sending " + messages.size() + " messages to server"));
        boolean success = this.sendDataToServer(out -> {
            try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)out, StandardCharsets.UTF_8);){
                this.writeMessagesToJson(writer, messages);
            }
        });
        LogLog.debug((String)"Messages sent");
        if (success && this.backupFileExists) {
            this.sendBackupFileToServer();
        }
        return success;
    }

    private void sendBackupFileToServer() {
        LogLog.debug((String)"Sending backup file to server");
        boolean success = this.sendDataToServer(out -> {
            try (FileChannel backupFileChannel = FileChannel.open(this.backupFile, new OpenOption[0]);
                 WritableByteChannel outChannel = Channels.newChannel(out);){
                long fileSize;
                long bytesLeft = fileSize = Files.size(this.backupFile);
                while ((bytesLeft -= backupFileChannel.transferTo(fileSize - bytesLeft, bytesLeft, outChannel)) > 0L) {
                }
            }
        });
        if (success) {
            try {
                Files.delete(this.backupFile);
                this.backupFileExists = false;
            }
            catch (IOException e) {
                String msg = "Error deleting backup file: ";
                LoggerFactory.getLogger(LogServerConnector.class).error(msg + e);
                LogLog.error((String)msg, (Throwable)e);
            }
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private boolean sendDataToServer(ThrowingConsumer<OutputStream> dataGenerator) {
        try {
            HttpsURLConnection cnn = (HttpsURLConnection)new URL(this.serverSettings.serverUrl).openConnection();
            if (this.serverSettings.allowSelfSignedCertificate) {
                Objects.requireNonNull(this.trustAllCerts, "trustAllCerts cannot be null when allowSelfSignedCertificate is enabled");
                SSLContext sc = SSLContext.getInstance("SSL");
                sc.init(null, this.trustAllCerts, null);
                cnn.setSSLSocketFactory(sc.getSocketFactory());
            }
            cnn.setUseCaches(false);
            cnn.setInstanceFollowRedirects(false);
            cnn.setConnectTimeout(10000);
            cnn.setReadTimeout(60000);
            cnn.setRequestMethod("POST");
            cnn.setDoOutput(true);
            cnn.setRequestProperty("Content-Type", "application/stream+json");
            cnn.setRequestProperty("Content-Encoding", "gzip");
            cnn.setRequestProperty("X-Installation-ID", this.serverSettings.instalaltionId);
            cnn.setRequestProperty("X-Installation-ID-Psw", this.serverSettings.password);
            try {
                cnn.connect();
                try (BufferedOutputStream out = new BufferedOutputStream(new GZIPOutputStream(cnn.getOutputStream(), 10240), 10240);){
                    dataGenerator.accept(out);
                }
                int code = cnn.getResponseCode();
                if (code != 202) throw new IllegalStateException("Received unexpected response code from the server. Code: " + code);
                this.connectionErrorStarted = -1L;
                boolean bl = true;
                return bl;
            }
            finally {
                cnn.disconnect();
            }
        }
        catch (Exception e) {
            if (this.connectionErrorStarted == -1L) {
                this.connectionErrorStarted = System.currentTimeMillis();
            }
            String msg = "Error sending data to log server: ";
            LoggerFactory.getLogger(LogServerConnector.class).error(msg + e);
            LogLog.error((String)msg, (Throwable)e);
            return false;
        }
    }

    private boolean writeMessagesToBackupFile(Collection<LogServerMessage> messages) {
        LogLog.debug((String)"Writing messages to backup file");
        try {
            if (!this.backupFileExists && Files.notExists(this.backupFile, new LinkOption[0]) && Files.notExists(this.backupFile.getParent(), new LinkOption[0])) {
                Files.createDirectories(this.backupFile.getParent(), new FileAttribute[0]);
            }
            try (BufferedWriter out = Files.newBufferedWriter(this.backupFile, StandardCharsets.UTF_8, StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.APPEND);){
                this.writeMessagesToJson(out, messages);
            }
            this.backupFileExists = true;
            return true;
        }
        catch (IOException e) {
            String msg = "Error saving messages to log file: ";
            LoggerFactory.getLogger(LogServerConnector.class).error(msg + e);
            LogLog.error((String)msg, (Throwable)e);
            return false;
        }
    }

    private final <T extends LogServerMessageBase> void writeMessagesToJson(Writer out, Collection<T> messages) throws IOException {
        DateTimeFormatter dtFormatter = LogServerConnector.initFormatter();
        for (LogServerMessageBase message : messages) {
            JsonWriter jsonWriter = new JsonWriter(out);
            jsonWriter.beginObject();
            message.writeFields(jsonWriter, dtFormatter);
            jsonWriter.endObject();
            out.write(10);
        }
    }

    void stop() {
        if (this.serverSettings == null) {
            return;
        }
        this.senderThread.interrupt();
        try {
            this.senderThread.join(10000L);
        }
        catch (InterruptedException e) {
            LogLog.warn((String)"LogServerConnector interrupted while waiting for sender thread to complete", (Throwable)e);
        }
        this.messagesQueue.clear();
    }

    private static final DateTimeFormatter initFormatter() {
        return DateTimeFormatter.ofPattern("yyyy-MM-dd'T'HH:mm:ss.SSSZZ").withZone(ZoneId.systemDefault());
    }

    private final String generateMessageId() {
        return Environment.getApplicationId() + '-' + (this.serverSettings != null ? this.serverSettings.instalaltionId : "null_settings") + '-' + UUID.randomUUID().toString();
    }

    @FunctionalInterface
    private static interface ThrowingConsumer<T> {
        public void accept(T var1) throws Exception;
    }
}

