package com.gridnine.xtrip.server.ibecorp.tasks.groovy.ibecorp7183

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.midoffice.helper.ReservationGdsNameInfoHelper
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.Xeption
import com.gridnine.xtrip.common.model.booking.BookingFile
import com.gridnine.xtrip.common.model.booking.BookingFileIndex
import com.gridnine.xtrip.common.model.booking.ProductStatus
import com.gridnine.xtrip.common.model.booking.RecordLocatorType
import com.gridnine.xtrip.common.model.booking.Reservation
import com.gridnine.xtrip.common.model.booking.ReservationType
import com.gridnine.xtrip.common.model.booking.xtriphotels.HotelProduct
import com.gridnine.xtrip.common.model.booking.xtriphotels.HotelProvider
import com.gridnine.xtrip.common.model.booking.xtriphotels.MainHotelProductIndex
import com.gridnine.xtrip.common.model.dict.GdsName
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.helpers.BookingHelper
import com.gridnine.xtrip.common.model.helpers.BookingStreamHelper
import com.gridnine.xtrip.common.model.helpers.HotelProductHelper
import com.gridnine.xtrip.common.model.profile.Person
import com.gridnine.xtrip.common.model.system.BaseFileDocument
import com.gridnine.xtrip.common.model.system.BasicDocumentIndex
import com.gridnine.xtrip.common.model.system.DocumentType
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

import java.util.stream.Collectors

@Field String fileSuffix = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date());
@Field FastDateFormat DTF = FastDateFormat.getInstance("dd.MM.yyyy HH:mm");
@Field FastDateFormat DF = FastDateFormat.getInstance("dd.MM.yyyy");
@Field EntityStorage entityStorage = EntityStorage.get()

@Field boolean onlyLook = true;
@Field boolean stopWhenError = true;
@Field Date startDate = MiscUtil.createDate(2024, 6, 1);
@Field Date endDate = MiscUtil.createDate(2024, 6, 12);
@Field EntityReference<BookingFile> bookingRef = null
//@Field EntityReference<BookingFile> bookingRef = new EntityReference("640830d8-3e92-4db1-9ae9-e152a56d0055", BookingFile.class, null)

doJob()

void doJob() {
    if (onlyLook){
        fileMessage("Режим 'Только подсчет'");
    }
    fileMessage("Останавливать процесс при ошибке? ${stopWhenError ? 'Да' : 'Нет'}");


    SearchQuery query = new SearchQuery();
    List<SearchCriterion> criterions = query.getCriteria().getCriterions();
    criterions.add(SearchCriterion.or(
            SearchCriterion.eq(
                    MainHotelProductIndex.Property.gdsName.name(),
                    GdsName.HOTEL_AGGREGATOR),
            SearchCriterion.eq(
                    MainHotelProductIndex.Property.reservationType.name(),
                    ReservationType.CORTEOS)));
    criterions.add(SearchCriterion.not(
            SearchCriterion.eq(MainHotelProductIndex.Property.provider.name(),
                    HotelProvider.INTERNAL)));
    criterions.add(SearchCriterion.not(
            SearchCriterion.in(
                    MainHotelProductIndex.Property.status.name(),
                    ProductStatus.INTENTION, ProductStatus.VOID_INTENTION)));
    if (bookingRef != null) {
        EntityContainer<BookingFile> bookingCtr = entityStorage.resolve(bookingRef)
        if (bookingCtr == null) {
            throw Xeption.forEndUser("Невалидная ссылка на заказ {0}", bookingRef.getUid());
        }
        fileMessage("Фильтр по заказу ${bookingCtr.entity.toString()}")
        criterions.add(
                SearchCriterion.eq("containerUid", bookingCtr.uid))
    } else {
        fileMessage("Фильтр по дате создания заказа: ${DF.format(startDate)}-${DF.format(endDate)}")
        criterions.add(
                SearchCriterion.ge(MainHotelProductIndex.Property.bookingCreationDate.name(), startDate))
        criterions.add(
                SearchCriterion.le(MainHotelProductIndex.Property.bookingCreationDate.name(), endDate))
    }

    query.getPreferredProperties().add(MainHotelProductIndex.Property.bookingNumber.name());

    List<MainHotelProductIndex> indexes = entityStorage.search(MainHotelProductIndex.class, query).getData();
    fileMessage("");
    fileMessage("Найдено ${indexes.size()} отельных продуктов");

    List<EntityReference<BookingFile>> bookingRefs = indexes.collect { it.source }.unique().collect()
    fileMessage("Найдено ${bookingRefs.size()} заказов");
    fileMessage("");
    fileMessage("");

    def total = bookingRefs.size()
    def current = 0
    def progr = 0
    int errors = 0
    Set<String> changedBookings = new HashSet<>();
    Set<String> unChangedBookings = new HashSet<>();
    int deleted = 0

    for (EntityReference<BookingFile> ref : bookingRefs) {
        if (isToBeStopped()) {
            fileMessage("iterator stopped")
            break;
        }

        try {
            EntityContainer<BookingFile> bookingCtr = entityStorage.resolve(ref);
            List<Reservation> reservations = BookingHelper.getReservationsForProduct(bookingCtr.getEntity(), HotelProduct.class);
            reservations.removeAll { getCheckOutDate(it) == null }
            if (reservations.size() < 2) {
                // баг был в случае нескольких отельных бронирований в заказе
                unChangedBookings.add(ref.getCaption())
            } else {
                Map<String, Date> entries = reservations.collectEntries { [ReservationGdsNameInfoHelper.getOnlineRecordLocator(it, RecordLocatorType.HOTEL_AGGREGATOR_PNR), getCheckOutDate(it)] }
                //fileMessage("${ref.caption}: ${entries.inspect()}")
                Date minCheckOutDate = entries.collect { it.value }.findAll { it != null }.min()

                SearchQuery docsQuery = new SearchQuery();
                docsQuery.getCriteria().getCriterions().add(
                        SearchCriterion.eq(BasicDocumentIndex.Property.owner.name(), ref))
                docsQuery.getCriteria().getCriterions().add(
                        SearchCriterion.eq(BasicDocumentIndex.Property.type.name(), DocumentType.GDS_LOG))
                docsQuery.getCriteria().getCriterions().add(
                        SearchCriterion.startsWith(BasicDocumentIndex.Property.name.name(), "sync-reservation"))
                docsQuery.getCriteria().getCriterions().add(
                        SearchCriterion.ge(BasicDocumentIndex.Property.created.name(), MiscUtil.addDaysToDate(minCheckOutDate, 1)))
                docsQuery.preferredProperties.add(BasicDocumentIndex.Property.name.name())
                docsQuery.preferredProperties.add(BasicDocumentIndex.Property.created.name())
                List<BasicDocumentIndex> docsIndexes =
                        entityStorage.search(BasicDocumentIndex.class, docsQuery).getData()
                if (docsIndexes.isEmpty()) {
                    // лишних документов нет
                    unChangedBookings.add(ref.getCaption())
                } else {
                    //fileMessage("Документы заказа ${ref.caption}: ${docsIndexes.size()}")
                    Map<String, EntityReference<BookingFile>> newBookingsOfPnr = new HashMap<>();
                    for (BasicDocumentIndex docInd : docsIndexes) {
                        EntityContainer<BaseFileDocument> docCtr = entityStorage.resolve(docInd.source)
                        String content = new String(docCtr.getEntity().getContent(), "utf-8")
                        boolean processed = false
                        for (Map.Entry<String, Date> entry : entries.entrySet()) {
                            String pnr = entry.key
                            checkOutDate = entry.value
                            if (content.contains("<pnr>${pnr}</pnr>")) {
                                processed = true
                                if (docCtr.entity.created.after(MiscUtil.addDaysToDate(checkOutDate, 1))) {
                                    if (!onlyLook) {
                                        entityStorage.delete(docCtr)
                                    }
                                    deleted++
                                    changedBookings.add(ref.getCaption())
                                }
                                break;
                            }
                        }
                        if (!processed) {
                            String pnr = content.find("<pnr>.*</pnr>")?.replaceAll("</?pnr>","")
                            //fileMessage("PNR of document: ${pnr}");
                            EntityReference<BookingFile> newBookingRef = newBookingsOfPnr.get(pnr)
                            if (newBookingRef == null) {
                                newBookingRef = findBookingByPnr(pnr);
                                newBookingsOfPnr.put(pnr, newBookingRef)
                                if (newBookingRef != null) {
                                    fileMessage("Документы бронирования ${pnr} были перенесены из заказа ${ref.caption} в ${newBookingRef.caption}");
                                }
                            }
                            if (newBookingRef != null) {
                                if (!onlyLook){
                                    docCtr.getEntity().setOwner(newBookingRef)
                                    docCtr.getVersionInfo().setDataSource("IBECORP-7183");
                                    docCtr.getVersionInfo().setVersionNotes("change owner");
                                    entityStorage.save(docCtr, true)
                                }
                            } else {
                                throw new MyException("Документ ${docCtr.entity.toString()} (${docCtr.uid}): не определено бронирование")
                            }
                        }
                    }
                }
            }
        } catch (MyException ex) {
            errors++;
            fileMessage("Ошибка (${ref.caption} / ${ref.uid}) : ${ex.message}");
            handleException(ex, ref)
        } catch (Exception ex) {
            errors++;
            fileMessage("Ошибка (${ref.caption} / ${ref.uid}) : ${TextUtil.getExceptionStackTrace(ex)}", false);
            handleException(ex, ref)
        }

        def progrNew = Math.round(current * 100d / total)
        if (progr != progrNew) {
            progress("${current + 1} / ${total}".toString(), (int) progrNew)
        }
        progr = progrNew
        current++
    }
    def progrNew = Math.round(current * 100d / total)
    progress("${current} / ${total}".toString(), (int) progrNew)

    fileMessage("Обработано заказов: ${current}");
    fileMessage("Удалено документов: ${deleted}");
    fileMessage("Заказы с удаленными документами: ${changedBookings.size()}");
    fileMessage(changedBookings.sort().toString(), false);
    fileMessage("", false)
    fileMessage("Заказы с неудаленными документами: ${unChangedBookings.size()}");
    //fileMessage(unChangedBookings.sort().toString(), false);
    fileMessage("", false)
    fileMessage("Возникло ошибок: ${errors}");
}

Date getCheckOutDate(Reservation res) {
    List<HotelProduct> products = BookingStreamHelper.getProductsStream(res, HotelProduct.class).collect(Collectors.toList())
    return products.collect { HotelProductHelper.getCheckOutDate(it) }.findAll { it != null }.min()
}

void handleException(Exception ex, EntityReference<Person> ref) {
    if (stopWhenError) {
        fileMessage("")
        fileMessage("Процесс остановлен из-за ошибки при обработке заказа ${ref.caption} / ${ref.uid}");
        throw new Exception("Процесс остановлен из-за ошибки при обработке заказа ${ref.caption} / ${ref.uid}", ex);
    }
}

EntityReference<BookingFile> findBookingByPnr(String pnr) {
    if (TextUtil.isBlank(pnr)){
        return null;
    }
    SearchQuery query = new SearchQuery();
    List<SearchCriterion> criterions = query.getCriteria().getCriterions();
    criterions.add(
            SearchCriterion.contains(BookingFileIndex.Property.pnr.name(), pnr))
    query.getPreferredProperties().add(BookingFileIndex.Property.fullNumber.name())
    List<BookingFileIndex> indexes = entityStorage.search(BookingFileIndex.class, query).getData()
    return indexes.size() == 1 ? indexes.get(0).getSource() : null
}

class MyException extends Exception {

    MyException(final String var) {
        super(var)
    }
}

void fileMessage(String str) {
    fileMessage(str, true)
}

void fileMessage(String str, boolean writeToConsole) {
    if (writeToConsole) {
        message(str);
    }
    String suffix = onlyLook ? "-onlyLook" : ""
    new File(Environment.getTempFolder(), "ibecorp7183${suffix}-${fileSuffix}.txt")
            .append("${DTF.format(new Date())}: ${str} \n");
}
