package com.gridnine.xtrip.server.ibecorp.tasks.groovy.ibecorp7273

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.profile.EmployeeCategory
import com.gridnine.xtrip.common.model.profile.Organization
import com.gridnine.xtrip.common.model.profile.Person
import com.gridnine.xtrip.common.model.profile.PersonIndex
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.search.SortOrder
import com.gridnine.xtrip.common.search.utils.SearchQueryHelper
import com.gridnine.xtrip.common.util.CollectionUtil
import com.gridnine.xtrip.common.util.LocaleUtil
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field String fileSuffix = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date());
@Field FastDateFormat DTF = FastDateFormat.getInstance("dd.MM.yyyy HH:mm");
@Field EntityStorage entityStorage = EntityStorage.get()

@Field boolean onlyLook = true;
@Field boolean stopWhenError = true;
@Field int limit = 0;
@Field int updateLimit = 1000;

@Field List<EntityReference<Organization>> orgRefs = null;
//@Field List<EntityReference<Organization>> orgRefs = Arrays.asList(
//        new EntityReference<>("8b280a2d-6a22-4415-8571-86ac33f36be0", Organization.class, null),
//        new EntityReference<>("83779a60-0f85-4feb-a77c-af9de5a64da4", Organization.class, null),
//        new EntityReference<>("18ab1640-89d2-4e4a-9363-88ac91bd42b0", Organization.class, null),
//        new EntityReference<>("b799ec06-7d43-4d9c-8bd3-bbb81e86f5c7", Organization.class, null),
//        new EntityReference<>("7790b178-42e5-4078-82df-6841cfd75e69", Organization.class, null),
//        new EntityReference<>("1badf562-0256-439e-a088-4a61bb75b01b", Organization.class, null),
//        new EntityReference<>("60534773-9d97-4d09-8946-938b64bb5eba", Organization.class, null),
//        new EntityReference<>("c4fba469-e4c0-488c-8e4d-edcb9021cac1", Organization.class, null),
//        new EntityReference<>("646314bc-8f8d-4cdc-aa31-b8c853d05b54", Organization.class, null),
//        new EntityReference<>("7ba0c653-b914-4f69-a4d2-66058bf348e7", Organization.class, null),
//)

if (onlyLook){
    fileMessage("Режим 'Только подсчет'");
}
doJob()


void doJob(){
    fileMessage("Останавливать процесс при ошибке? ${stopWhenError ? 'Да' : 'Нет'}");
    fileMessage("");

    SearchQuery query = new SearchQuery();
    query.getCriteria().getCriterions().add(
            SearchCriterion.isNotEmpty(PersonIndex.Property.employeeCategoriesNames.name()));

    if (CollectionUtil.isNotEmpty(orgRefs)) {
        List<String> orgNames = new ArrayList<>()
        orgRefs.each {
            EntityContainer<Organization> orgCtr = EntityStorage.get().resolve(it);
            if (orgCtr == null) {
                throw Xeption.forEndUser("Невалидная ссылка на организацию {0}", orgRef.getUid());
            }
            orgNames.add(orgCtr.getEntity().toString())
        }
        fileMessage("Фильтр по организациям ${orgNames}");
        query.getCriteria().getCriterions().add(
                SearchQueryHelper.buildOrContainsEntityCriterion(PersonIndex.Property.employmentsUids.name(), orgRefs))
    }

    query.getPreferredProperties().add("containerUid");
    query.getPreferredProperties().add(PersonIndex.Property.company.name());
    query.getCriteria().getOrders().put("companyReferenceCaption", SortOrder.ASC)
    if (!onlyLook){
        if (limit > 0) {
            query.setLimit(limit)
            fileMessage("Установлено ограничение на поиск $limit");
        }
        fileMessage("Установлено ограничение на изменение $updateLimit");
    }

    List<PersonIndex> pspIndices = EntityStorage.get().search(PersonIndex.class, query).getData();

    fileMessage("Найдено ${pspIndices.size()} физ. лиц с категориями");

    if (pspIndices.isEmpty()) {
        return;
    }

    def total = pspIndices.size()
    def current = 0
    def progr = 0
    int errors = 0
    int changed = 0;
    int unChanged = 0;

    fileMessage("");
    fileMessage("");

    Map<EntityReference, List<PersonIndex>> groupedIndices = pspIndices.groupBy { it.company }

    groupedIndices.each {oRef, list ->
        if (updateLimit != 0 && changed > updateLimit) {
            return;
        }
        if (oRef==null){
            fileMessage("Обработка ${list.size()} пассажиров без основной организации");
        } else {
            fileMessage("Обработка ${list.size()} пассажиров основной организации ${oRef.caption} (${oRef.uid})");
        }
        Map<EntityReference<Organization>, EntityContainer<Organization>> resolvedOrgs = new HashMap<>()
        resolvedOrgs.put(oRef, entityStorage.resolve(oRef));
        List<String> changedPersons = new ArrayList<>();
        List<String> unChangedPersons = new ArrayList<>();
        int errorsByOrg = 0;
        for (PersonIndex ind : list) {
            if (isToBeStopped()) {
                fileMessage("iterator stopped")
                break;
            }
            if (updateLimit != 0 && changed > updateLimit) {
                fileMessage("Обработка завершена в соответствии с лимитом ${updateLimit}")
                break;
            }
            EntityReference<Person> ref =  ind.source
            try {
                EntityContainer<Person> personCtr = entityStorage.resolve(ref);
                if (updated(personCtr.getEntity(), resolvedOrgs)){
                    if (!onlyLook){
                        personCtr.getVersionInfo().setDataSource("IBECORP-7273");
                        personCtr.getVersionInfo().setVersionNotes("Исправление категорий организаций");
                        entityStorage.save(personCtr, true);
                    }
                    changed++
                    changedPersons.add("{${personCtr.getEntity().toString()} (${personCtr.toReference().getUid()})}");
                } else {
                    unChanged++
                    unChangedPersons.add("{${personCtr.getEntity().toString()} (${personCtr.toReference().getUid()})}");
                }
            } catch (MyException ex) {
                errorsByOrg++
                errors++;
                fileMessage("Ошибка (${ref.caption} / ${ref.uid}) : ${ex.message}");
                handleException(ex, ref)
            } catch (Exception ex) {
                errorsByOrg++
                errors++;
                fileMessage("Ошибка (${ref.caption} / ${ref.uid}) : ${TextUtil.getExceptionStackTrace(ex)}", false);
                handleException(ex, ref)
            }

            def progrNew = Math.round(current * 100d / total)
            if (progr != progrNew) {
                progress("${current + 1} / ${total}".toString(), (int) progrNew)
            }
            progr = progrNew
            current++
        }
        fileMessage("Изменено ${changedPersons.size()}, Не изменено ${unChangedPersons.size()}, Ошибок ${errorsByOrg}");
        fileMessage("Список измененных пассажиров : ${changedPersons.toString()}", false);
        //fileMessage("Список неизмененных пассажиров : ${unChangedPersons.toString()}", false);
        fileMessage("");
        fileMessage("");
    }
    def progrNew = Math.round(current * 100d / total)
    progress("${current} / ${total}".toString(), (int) progrNew)

    fileMessage("Изменено пассажиров: ${changed}");
    fileMessage("Не изменено пассажиров: ${unChanged}");
    fileMessage("Возникло ошибок: ${errors}");
}

void handleException(Exception ex, EntityReference<Person> ref){
    if (stopWhenError){
        fileMessage("Процесс остановлен из-за ошибки в профиле ${ref.caption} / ${ref.uid}");
        throw new Exception("Процесс остановлен из-за ошибки в профиле ${ref.caption} / ${ref.uid}", ex);
    }
}

boolean updated(Person person, Map<EntityReference<Organization>, EntityContainer<Organization>> resolvedOrgs){
    boolean updated = false
    person.getEmployments().each {emp ->
        def empCategory = emp.getEmployeeCategory();
        if (empCategory != null && empCategory.getName()!=null) {
            def orgRef = emp.getOrganization()
            EntityContainer<Organization> orgCtr = null
            if (!resolvedOrgs.containsKey(orgRef)){
                orgCtr = entityStorage.resolve(orgRef)
                resolvedOrgs.put(orgRef, orgCtr)
            } else {
                orgCtr = resolvedOrgs.get(orgRef)
            }
            if (orgCtr != null && orgCtr.getEntity()!=null) {
                EmployeeCategory orgEmpCategory = findEmployeeCategory(orgCtr.getEntity(), empCategory)
                if (orgEmpCategory==null){
                    throw new MyException("У оргинзации ${orgCtr.getEntity().toString()} не найдена категория ${empCategory.getName().toString(LocaleUtil.LOCALE_RU)}");
                }
                if (!MiscUtil.equals(orgEmpCategory.uid, empCategory.uid)){
                    if (!onlyLook){         // проверять обязательно, иначе закэшируется несохраненное значение!
                        emp.setEmployeeCategory(orgEmpCategory)
                    }
                    updated = true
                }
            }
        }
    }
    return updated
}

EmployeeCategory findEmployeeCategory(final Organization org, final EmployeeCategory category) {
    def values = category.getName().getValues().values();
    return org.getEmployeeCategories().find {c ->
        c.name!=null && CollectionUtil.containAtLeastOneCommonElement(c.name.values.values(), values)
    }
}

class MyException extends Exception {

    MyException(final String var) {
        super(var)
    }
}

void fileMessage(String str) {
    fileMessage(str, true)
}

void fileMessage(String str, boolean writeToConsole) {
    if (writeToConsole) {
        message(str);
    }
    String suffix = onlyLook ? "-onlyLook" : ""
    new File(Environment.getTempFolder(), "ibecorp7273${suffix}-${fileSuffix}.txt")
            .append("${DTF.format(new Date())}: ${str} \n");
}