package com.gridnine.xtrip.server.ibecorp.tasks.groovy.uts723

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.booking.BookingFile
import com.gridnine.xtrip.common.model.booking.xtriphotels.HotelProduct
import com.gridnine.xtrip.common.model.booking.xtriphotels.MainHotelProductIndex
import com.gridnine.xtrip.common.model.dict.CodeSystem
import com.gridnine.xtrip.common.model.dict.DictionaryCache
import com.gridnine.xtrip.common.model.dict.GeoLocation
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.entity.parameters.EntityStorageSaveParameters
import com.gridnine.xtrip.common.model.helpers.BookingHelper
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.util.MiscUtil
import com.gridnine.xtrip.common.util.TextUtil
import com.gridnine.xtrip.server.model.tasks.standard.ExecuteGroovyScriptTask
import com.gridnine.xtrip.server.storage.DictionaryStorage
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field String fileSuffix = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date());
@Field FastDateFormat DTF = FastDateFormat.getInstance("dd.MM.yyyy HH:mm");

@Field DictionaryCache dictCache = DictionaryCache.get();
@Field DictionaryStorage dictStorage = DictionaryStorage.get();
@Field EntityStorage entityStorage = EntityStorage.get();

@Field boolean stopWhenError = true;
@Field boolean onlyLook = true;


doJob();

void doJob() {
    if (onlyLook) {
        fileMessage("Режим 'Только просмотр'")
    }
    fileMessage("Останавливать процесс при ошибке? ${stopWhenError ? 'Да' : 'Нет'}")
    fileMessage("");

    ExecuteGroovyScriptTask.InputFile file = inputFile;
    if (file == null) {
        fileMessage("no file");
        return;
    }
    String fileData = new String(file.getBody(), "utf-8");
    if (TextUtil.isBlank(fileData)) {
        fileMessage("Список уидов документов пуст");
        return;
    }
    String[] rows = fileData.split("\n");

    def current = 0
    def total = rows.size()
    def progr = 0
    int errors = 0
    int locationsIgnored = 0
    int locationsDeleted = 0
    int locationsUpdated = 0
    int bookingsUpdated = 0

    fileMessage("Будет обработано $total записей");

    if (total == 0) {
        return;
    }

    for (String row : rows) {
        if (isToBeStopped()) {
            fileMessage("iterator stopped")
            break;
        }
        try {
            if (TextUtil.nonBlank(row)) {
                String codesData = row.split(";")[0]
                String[] codes = codesData.split(" -> ")
                String codeFrom = codes[0]
                String codeTo = codes[1]

                Set<GeoLocation> locationsFrom = findLocations(codeFrom)
                if (locationsFrom.isEmpty()) {
                    def progrNew = Math.round(current * 100d / total)
                    if (progr != progrNew) {
                        progress("${current + 1} / ${total}".toString(), (int) progrNew)
                    }
                    progr = progrNew
                    current++

                    fileMessage("${codesData}: Не найдена локация с ОА-кодом ${codeFrom}", false);
                    locationsIgnored++
                    continue;
                }
                if (locationsFrom.size() > 1) {
                    throw new MyException("found many locations with ha-code ${codeFrom}");
                }
                GeoLocation locationFrom = locationsFrom.iterator().next()

                Set<GeoLocation> locationsTo = findLocations(codeTo)
                if (locationsTo.isEmpty()) {
                    if (!onlyLook) {
                        locationFrom.getCodeVariants().put(CodeSystem.HOTEL_AGGREGATOR.name(), codeTo)
                        locationFrom.setDataSource("UTS-723")
                        locationFrom.setDataSource("update HOTEL_AGGREGATOR code")
                        dictStorage.save(locationFrom)
                    }
                    locationsUpdated++
                    fileMessage("${codesData}: ${locationFrom.code} (${codeFrom}) изменен код ОА", false);
                } else {
                    if (locationsTo.size() > 1) {
                        throw new MyException("found many locations with ha-code ${codeTo}");
                    }
                    GeoLocation locationTo = locationsTo.iterator().next()

                    SearchQuery query = new SearchQuery();
                    query.getCriteria().getCriterions().add(
                            SearchCriterion.eq(MainHotelProductIndex.Property.hotelCity.name(), locationFrom.toReference()))
                    query.getPreferredProperties().add("containerUid");
                    List<MainHotelProductIndex> productIndices = entityStorage.search(MainHotelProductIndex.class, query).getData()
                    List<EntityReference<BookingFile>> bookingRefs = productIndices.collect { it.source }.unique()
                    fileMessage("${codesData}: найдено ${bookingRefs.size()} заказов с локацией ${locationFrom.code}", false);
                    if (!bookingRefs.isEmpty()) {
                        fileMessage("${bookingRefs.collect { it.caption }}", false);
                    }
                    for (EntityReference<BookingFile> bookingRef : bookingRefs) {
                        EntityContainer<BookingFile> bookingCtr = entityStorage.resolve(bookingRef)
                        List<HotelProduct> products = BookingHelper.getProductsList(bookingCtr.entity, HotelProduct.class).findAll {
                            MiscUtil.equals(it.hotelLocation, locationFrom.toReference())
                        }
                        if (products.isEmpty()) {
                            fileMessage("${codesData}: в заказе ${bookingRef.caption} не найдены продукты с локацией ${locationFrom.code}", false);
                        } else {
                            if (!onlyLook) {
                                products.each {
                                    it.setHotelLocation(locationTo.toReference())
                                }
                                bookingCtr.getVersionInfo().setDataSource("UTS-723")
                                bookingCtr.getVersionInfo().setVersionNotes("update hotelLocation ${codesData}")
                                entityStorage.save(bookingCtr, true, new EntityStorageSaveParameters().ignoreInterceptors(true))
                            }
                            bookingsUpdated++
                        }
                    }

                    if (locationFrom.getCodeVariants().size() == 1 && locationFrom.getCodeVariants().keySet().contains(CodeSystem.HOTEL_AGGREGATOR.name())) {
                        if (!onlyLook) {
                            dictStorage.delete(locationFrom)
                        }
                        locationsDeleted++
                        fileMessage("${codesData}: ${locationFrom.code} (${codeFrom}) удален", false);
                    } else {
                        if (!onlyLook) {
                            locationFrom.getCodeVariants().remove(CodeSystem.HOTEL_AGGREGATOR.name())
                            locationFrom.setDataSource("UTS-723")
                            locationFrom.setDataSource("delete HOTEL_AGGREGATOR code")
                            dictStorage.save(locationFrom)
                        }
                        locationsUpdated++
                        fileMessage("${codesData}: ${locationFrom.code} (${codeFrom}) удален код ОА", false);
                    }
                }
            }
        } catch (MyException ex) {
            errors++;
            fileMessage("Ошибка (${row.trim()}) : ${ex.message}");
            if (stopWhenError) {
                fileMessage("Процесс остановлен из-за ошибки в записи ${row.trim()}");
                current++
                break;
            }
        } catch (Exception ex) {
            errors++;
            fileMessage("Ошибка (${row.trim()}) : ${TextUtil.getExceptionStackTrace(ex)}", false);
            if (stopWhenError) {
                fileMessage("Процесс остановлен из-за ошибки в записи ${row.trim()}");
                current++
                break;
            }
        }
        def progrNew = Math.round(current * 100d / total)
        if (progr != progrNew) {
            progress("${current + 1} / ${total}".toString(), (int) progrNew)
        }
        progr = progrNew
        current++
    }
    def progrNew = Math.round(current * 100d / total)
    progress("${current} / ${total}".toString(), (int) progrNew)

    fileMessage("");
    fileMessage("Обработано: ${current}");
    fileMessage("");
    fileMessage("Возникло ошибок: ${errors}");
    fileMessage("Пропущено локаций: ${locationsIgnored}");
    fileMessage("Удалено локаций: ${locationsDeleted}");
    fileMessage("Изменено локаций: ${locationsUpdated}");
    fileMessage("");
    fileMessage("Изменено заказов: ${bookingsUpdated}");

}

Set<GeoLocation> findLocations(String code) {
    Set<GeoLocation> result = dictCache.lookup(GeoLocation.class, code, CodeSystem.HOTEL_AGGREGATOR.name())
    return result.findAll { MiscUtil.equals(it.codeVariants.get(CodeSystem.HOTEL_AGGREGATOR.name()), code) }
}

class MyException extends Exception {

    MyException(final String var) {
        super(var)
    }
}

void fileMessage(String str) {
    fileMessage(str, true)
}

void fileMessage(String str, boolean writeToConsole) {
    if (writeToConsole) {
        message(str);
    }
    String suffix = onlyLook ? "-onlyLook" : ""
    new File(Environment.getTempFolder(), "uts723${suffix}-${fileSuffix}.txt")
            .append("${DTF.format(new Date())}: ${str} \n");
}