package com.gridnine.xtrip.server.ibecorp.tasks.groovy.xtr13016

import com.gridnine.xtrip.common.Environment
import com.gridnine.xtrip.common.l10n.model.L10nStringHelper
import com.gridnine.xtrip.common.model.EntityContainer
import com.gridnine.xtrip.common.model.EntityReference
import com.gridnine.xtrip.common.model.Xeption
import com.gridnine.xtrip.common.model.dict.PassportType
import com.gridnine.xtrip.common.model.entity.EntityStorage
import com.gridnine.xtrip.common.model.profile.*
import com.gridnine.xtrip.common.search.ResultMode
import com.gridnine.xtrip.common.search.SearchCriterion
import com.gridnine.xtrip.common.search.SearchQuery
import com.gridnine.xtrip.common.util.LocaleUtil
import com.gridnine.xtrip.common.util.TextUtil
import com.gridnine.xtrip.common.util.TranslitUtil
import groovy.transform.Field
import org.apache.commons.lang.time.FastDateFormat

@Field String fileSuffix = FastDateFormat.getInstance("yyyy.MM.dd_HH.mm").format(new Date());
@Field FastDateFormat DTF = FastDateFormat.getInstance("dd.MM.yyyy HH:mm");

@Field boolean onlyCount = true;
@Field boolean stopWhenError = true;
@Field int limit = 10;

@Field List<EntityReference<Organization>> orgRefs = null;
//@Field List<EntityReference<Organization>> orgRefs = Arrays.asList(
//        new EntityReference<>("a2316d5f-1879-4307-9126-1ab39ac74a21", Organization.class, null),
//        new EntityReference<>("fa27f2ea-639a-4e33-9298-70f31dbea9c7", Organization.class, null),
//        new EntityReference<>("f9a09727-39c4-44e0-9f4e-a7c19e977bde", Organization.class, null),
//)


doJob()


void doJob(){
    if (onlyCount){
        fileMessage("Режим 'Только подсчет'");
        calc()
    } else {
        update()
    }
}

void calc(){
    if (orgRefs == null) {
        calcPersons(null);
    } else {
        orgRefs.each {
            calcPersons(it);
            fileMessage("")
        }
    }
}

void calcPersons(EntityReference<Organization> orgRef){
    SearchQuery query = buildQuery(orgRef)
    query.setResultMode(ResultMode.COUNT_ONLY);
    int total = EntityStorage.get().search(PersonIndex.class, query).getTotalCount();

    fileMessage("Найдено $total путешественников без латинских фамилии и имени");
}

SearchQuery buildQuery(EntityReference<Organization> orgRef){
    SearchQuery query = new SearchQuery();
    query.getCriteria().getCriterions().add(
            SearchCriterion.contains(PersonIndex.Property.types.name(), PersonType.PASSENGER));
    query.getCriteria().getCriterions().add(
            SearchCriterion.isEmpty(PersonIndex.Property.englishLastName.name()));
    query.getCriteria().getCriterions().add(
            SearchCriterion.isEmpty(PersonIndex.Property.englishFirstName.name()));

    if (orgRef != null) {
        EntityContainer<Organization> orgCtr = EntityStorage.get().resolve(orgRef);
        if (orgCtr == null) {
            throw Xeption.forEndUser("Невалидная ссылка на организацию {0}", orgRef.getUid());
        }
        fileMessage("Фильтр по организации ${orgCtr.getEntity().toString()}");
        query.getCriteria().getCriterions().add(
                SearchCriterion.eq(PersonIndex.Property.company.name(), orgRef));
    }

    query.getPreferredProperties().add("containerUid");
    return query;
}

void update(){
    fileMessage("Установлено ограничение $limit");
    fileMessage("Останавливать процесс при ошибке? ${stopWhenError ? 'Да' : 'Нет'}");

    if (orgRefs == null) {
        updatePersons(null);
    } else {
        orgRefs.each {
            updatePersons(it);
            fileMessage("")
        }
    }
}

void updatePersons(EntityReference<Organization> orgRef) {
    if (onlyCount) {
        return
    }
    SearchQuery query = buildQuery(orgRef)
    query.setLimit(limit)

    List<OrganizationIndex> personIndices = EntityStorage.get().search(PersonIndex.class, query).getData();
    def current = 0
    def total = personIndices.size()
    def progr = 0
    int errors = 0

    fileMessage("Найдено $total путешественников");

    if (personIndices.isEmpty()) {
        return;
    }
    List<String> changedPersons = new ArrayList<>();
    List<String> unChangedPersons = new ArrayList<>();

    for (PersonIndex ind : personIndices) {
        if (isToBeStopped()) {
            fileMessage("iterator stopped")
            break;
        }
        try {
            EntityContainer<Person> personCtr = EntityStorage.get().resolve(ind.getSource());
            if (updated(personCtr.getEntity())){
                personCtr.getVersionInfo().setDataSource("XTR-13016");
                personCtr.getVersionInfo().setVersionNotes("Заполнение локали EN");
                EntityStorage.get().save(personCtr, true);
                changedPersons.add("{${personCtr.getEntity().toString()} (${personCtr.toReference().getUid()})}");
            } else {
                unChangedPersons.add("{${personCtr.getEntity().toString()} (${personCtr.toReference().getUid()})}");
            }
        } catch (Exception ex) {
            errors++;
            fileMessage("Ошибка (${ind.source.caption} / ${ind.source.uid}) : ${TextUtil.getExceptionStackTrace(ex)}", false);
            if (stopWhenError){
                fileMessage("Процесс остановлен из-за ошибки в профиле ${ind.source.caption} / ${ind.source.uid}");
                throw new Exception("Произошла ошибка при обработке пассажира", ex);
            }
        }

        def progrNew = Math.round(current * 100d / total)
        if (progr != progrNew) {
            progress("${current + 1} / ${total}".toString(), (int) progrNew)
        }
        progr = progrNew
        current++
    }
    def progrNew = Math.round(current * 100d / total)
    progress("${current} / ${total}".toString(), (int) progrNew)

    fileMessage("Возникло ошибок: ${errors}");
    fileMessage("Изменено пассажиров: ${changedPersons.size()}");
    fileMessage("Список измененных пассажиров : ${changedPersons.toString()}", false);
    fileMessage("Не изменено пассажиров: ${unChangedPersons.size()}");
    fileMessage("Список неизмененных пассажиров : ${unChangedPersons.toString()}", false);
}

boolean updated(Person person){
    String lastNameEn = L10nStringHelper.getValue(person.getLastName(), LocaleUtil.LOCALE_EN, true);
    String firstNameEn = L10nStringHelper.getValue(person.getFirstName(), LocaleUtil.LOCALE_EN, true);
    if (TextUtil.nonBlank(lastNameEn) || TextUtil.nonBlank(firstNameEn)){
        return false;
    }
    List<Passport> passports = new ArrayList(person.getPassports())
            .findAll { isPassportWithLatinNames(it)};
    passports.sort{a,b -> getPassportTypePriority(a) <=> getPassportTypePriority(b) }

    Passport psp = passports.isEmpty() ? null : passports.get(0)
    String docLastName = psp?.getLastName();
    String docFirstName = psp?.getFirstName();
    String docMiddleName = psp?.getMiddleName();

    if (updateNames(person, docLastName, docFirstName, docMiddleName)){
        return true;
    }

    String lastNameRu = L10nStringHelper.getValue(person.getLastName(), LocaleUtil.LOCALE_RU, true);
    String firstNameRu = L10nStringHelper.getValue(person.getFirstName(), LocaleUtil.LOCALE_RU, true);
    String middleNameRu = L10nStringHelper.getValue(person.getMiddleName(), LocaleUtil.LOCALE_RU, true);

    if (updateNames(person, lastNameRu, firstNameRu, middleNameRu)){
        // latin was in RU locale
        return true;
    }

    if (updateNames(person, cyr2lat(lastNameRu), cyr2lat(firstNameRu), cyr2lat(middleNameRu))){
        return true;
    }
    return false
}

String cyr2lat(String value) {
    return TranslitUtil.cyr2lat(value, TranslitUtil.TranslitRules.GOST_52535_1_2006);
}

boolean updateNames(Person person, String lastName, String firstName, String middleName ){
    boolean result = false
    if (TextUtil.nonBlank(lastName) || TextUtil.nonBlank(firstName)){
        if (isLatin(lastName)){
            L10nStringHelper.setValue(person.getLastName(), lastName, LocaleUtil.LOCALE_EN)
            result = true;
        }
        if (isLatin(firstName)){
            L10nStringHelper.setValue(person.getFirstName(), firstName, LocaleUtil.LOCALE_EN)
            result = true;
        }
        if (result && isLatin(middleName)){
            L10nStringHelper.setValue(person.getMiddleName(), middleName, LocaleUtil.LOCALE_EN)
        }
    }
    return result
}

boolean isPassportWithLatinNames(Passport psp){
    if (psp == null || psp.getType() == null) {
        return false;
    }
    return isLatin(psp.getLastName()) || isLatin(psp.getFirstName())
}

boolean isLatin(String str){
    if (TextUtil.isBlank(str)){
        return false
    }
    return str.matches('(^.*[A-Za-z])+.*$') && !TextUtil.containsCyrillicLetter(str)
}

int getPassportTypePriority(Passport psp){
    if (psp.getType()== PassportType.FOREIGN){
        return 0;
    }
    if (psp.getType()== PassportType.INTERNAL){
        return 1;
    }
    return 2;
}

void fileMessage(String str) {
    fileMessage(str, true)
}

void fileMessage(String str, boolean writeToConsole) {
    if (writeToConsole) {
        message(str);
    }
    String suffix = onlyCount ? "-onlyCount" : ""
    new File(Environment.getTempFolder(), "xtr13016${suffix}-${fileSuffix}.txt")
            .append("${DTF.format(new Date())}: ${str} \n");
}